#version 330

// use gl_FrontFacing to know if the fragment is front or back facing

smooth in vec2 texCoord;
smooth in vec4 theColor;
smooth in vec3 relativeCoord;

out vec4 outputColor;

uniform sampler2D colorTexture;
//uniform sampler2D fireTexture;

uniform vec4 skyColors[ 4 ];

uniform vec2 fogDistance;
uniform vec3 ambientColor;
uniform float saturation;

const float zNear = 0.1;
// see also startClient - TODO make it configurable
const float zFar = 4096.0;

#define M_PI 3.1415926535897932384626433832795

///////////////////////////////////////////////////////////////////////////////////////////

float depth2camera( float d )
{
	// [ 0, 1 ] -> [ -1, 1 ]
    float z_n = 2.0 * d - 1.0;
    float z_e = 2.0 * zNear * zFar / ( zFar + zNear - z_n * ( zFar - zNear ) );
	return z_e;
}

///////////////////////////////////////////////////////////////////////////////////////////

float fog_clamp( float dc, float dc_min, float dc_max )
{
	return ( clamp( dc, dc_min, dc_max ) - dc_min ) / ( dc_max - dc_min );
}

///////////////////////////////////////////////////////////////////////////////////////////

vec4 get_sky_color( vec2 current_radial_coord )
{
    vec4 horizontalColor = ( current_radial_coord.x >= 0.5 ) ?
        mix( skyColors[ 3 ], skyColors[ 1 ], ( current_radial_coord.x - 0.5 ) * 2 ) :       // side -> west
        mix( skyColors[ 2 ], skyColors[ 3 ], current_radial_coord.x * 2 );     // east -> side

    // Y: 0 : zenith, 1 : nadir
    float vertical_angle = 1 - current_radial_coord.y * 2;
    return mix( horizontalColor, skyColors[ 0 ], ( vertical_angle >= 0.0 ) ? vertical_angle : 0.0 );
}

///////////////////////////////////////////////////////////////////////////////////////////

// Unfortunately, this has to be done in the fragment shader. Otherwise, the interpolated radial
// coordinates based on the radial coordinates of the corners of the sky cube will be totally incorrect
// (some of them will be constant because the corners of the cube all have symmetrical coordinates).
vec2 get_radial_coord( vec3 current_relative_coord )
{
    vec2 radialCoord;
    float pos_diff_r = length( current_relative_coord );
    if( pos_diff_r < 0.01 )
    {
        radialCoord.x = 0.0;
        radialCoord.y = 0.0;
    }
    else
    {
        // 0 : EAST - direction towards positive X , PI -> 1.0 : WEST (negative X)
        radialCoord.x = acos( current_relative_coord.x / pos_diff_r ) / M_PI;

        // 0 : zenith - direction towards positive Y , PI -> 1.0 : nadir
        radialCoord.y = acos( current_relative_coord.y / pos_diff_r ) / M_PI;
    }

    return radialCoord;
}

///////////////////////////////////////////////////////////////////////////////////////////

void main()
{
    // Note: gl_FragCoord.xy are true window coordinates, not those of the viewport

    vec4 outputColorN = texture( colorTexture, texCoord ) * theColor ;

    // Discarding both color and depth information for transparent pixels.
    // Beware: this may impact performance as early depth tests become impossible (TBC).
    // This is needed to render objects beyond the objects that have transparent pixels.
    if( outputColorN.a < 0.75f )
    {
        discard;
 //       gl_FragDepth = 0.999f;
    }

    // Apply ambient color and saturation.
    // R'G'B' -> Y' BT.709 conversion
    float y = dot( vec3( 0.2126, 0.7152, 0.0722 ), outputColorN.rgb );
    vec4 yv = vec4( y, y, y, outputColorN.a );

    vec4 ambientv = vec4( ambientColor.r, ambientColor.g, ambientColor.b, 1.0 );

    // Desaturate first, add ambient color later.
    outputColorN = mix( yv, outputColorN, saturation ) * ambientv;

    // Eventually, we may wish to distinguish the rendered terrain + plants from the sky.
    // That would allow us to have cool effects like fogged out hill / tree silhouettes with the brightly
    // lit sunset behind them. Unfortunately, it is out of the question as long as we do not have larger
    // rendering distances (otherwise, we will see the said silhouettes pop in/out at the bright sky background).

    // Air fog
    // Note that zFar / 2 is the sky limit
    //float fog = fog_clamp( dc_opaque, 100, 2000 );
    float fog = fog_clamp( depth2camera( gl_FragCoord.z ), fogDistance.x, fogDistance.y );

    vec2 radialCoord = get_radial_coord( relativeCoord );
    vec4 skyColor = get_sky_color( radialCoord );

    outputColor = mix( outputColorN, skyColor, fog );
}
